/**
 *
 */
package gov.va.med.mhv.usermgmt.integration.service.util;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;

import org.apache.commons.lang.StringUtils;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.integration.registry.transfer.Address;
import gov.va.med.mhv.integration.registry.transfer.Alias;
import gov.va.med.mhv.integration.registry.transfer.ContactInformation;
import gov.va.med.mhv.integration.registry.transfer.Facility;
import gov.va.med.mhv.integration.registry.transfer.Gender;
import gov.va.med.mhv.integration.registry.transfer.Indicator;
import gov.va.med.mhv.integration.registry.transfer.Name;
import gov.va.med.mhv.integration.registry.transfer.Patient;
import gov.va.med.mhv.integration.registry.transfer.PersonalInfo;
import gov.va.med.mhv.integration.registry.transfer.PlaceOfBirth;
import gov.va.med.mhv.integration.registry.transfer.Status;
import gov.va.med.mhv.integration.registry.transfer.StatusType;
import gov.va.med.mhv.usermgmt.bizobj.FacilityInfoBO;
import gov.va.med.mhv.usermgmt.enumeration.CountryEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.GenderEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.StateEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.SuffixEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.TitleEnumeration;
import gov.va.med.mhv.usermgmt.transfer.FacilityInfo;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;
import gov.va.med.mhv.usermgmt.transfer.UserProfile;

/**
 * @author Rob Proper (Aquilent Inc.)
 *
 */
public final class RegistryInterfaceUtils {
    
    private static final DateFormat DATE_FORMAT = new SimpleDateFormat(
        "d-MMM-yyyy"); 

    private static final String UNKNOWN = "unknown";
    
    private RegistryInterfaceUtils() {
    }

    @SuppressWarnings("unchecked")
    public static Patient toPatient(
        gov.va.med.mhv.usermgmt.transfer.Patient patient)
    {
        Patient registryPatient = new Patient();
        registryPatient.setIcn(patient.getIcn());
        registryPatient.setUserId(patient.getUserProfile().getId());
        registryPatient.setPersonalInfo(toPersonalInfo(patient.
            getUserProfile()));
        registryPatient.setFacilities(toFacilities(patient.
            getFacilitys()));
        return registryPatient;
    }

    public static PersonalInfo toPersonalInfo(
        gov.va.med.mhv.usermgmt.transfer.Patient patient)
    {
        return toPersonalInfo(patient.getUserProfile());
    }

    public static PersonalInfo toPersonalInfo(
        UserProfile userProfile)
    {
        Precondition.assertNotNull("userprofile", userProfile);
        PersonalInfo personalInfo = new PersonalInfo();
        personalInfo.setName(toName(userProfile));
        personalInfo.setMothersMaidenName(null); // unknown in MHV
        personalInfo.setDateOfBirth(userProfile.getBirthDate());
        personalInfo.setPlaceOfBirth(null); // unknown in MHV
        personalInfo.setGender(toGender(userProfile.getGender()));
        personalInfo.setSsn(userProfile.getSsn());
        personalInfo.setMultipleBirthIndicator(null); // unknown in MHV
        personalInfo.setAlias(null); // unknown in MHV
        personalInfo.setEtnicity(null); // unknown in MHV
        personalInfo.setRace(null); // unknown in MHV
        personalInfo.setPermanentAddress(toAddress(userProfile));
        personalInfo.setContactInformation(toContactInformation(userProfile));
        return personalInfo;

    }

    public static Name toName(UserProfile userProfile) {
        Precondition.assertNotNull("userprofile", userProfile);
        Name name = new Name();
        name.setFirstName(userProfile.getFirstName());
        name.setMiddleName(userProfile.getMiddleName());
        name.setLastName(userProfile.getLastName());
        name.setPrefix((userProfile.getTitle() != null) 
            ? userProfile.getTitle().getName() : null);
        name.setSuffix((userProfile.getSuffix() != null) 
            ? userProfile.getSuffix().getName() : null);
        return name;
    }

    public static Address toAddress(UserProfile userProfile) {
        Precondition.assertNotNull("userprofile", userProfile);
        Address address = new Address();
        address.setStreetAddress1(userProfile.getAddressStreet1());
        address.setStreetAddress2(userProfile.getAddressStreet2());
        address.setStreetAddress3(null); // Unknown
        address.setCity(userProfile.getAddressCity());
        address.setStateOrProvince((userProfile.getAddressState() != null) 
            ? userProfile.getAddressState().getName()
            : userProfile.getAddressProvince());
        address.setZipOrPostalCode(userProfile.getAddressPostalCode());
        address.setCounty(null); // Unknown
        address.setCountry((userProfile.getAddressCountry() != null)
            ? userProfile.getAddressCountry().getName() : null);
        address.setBadAddressIndicator(null); // Unknown
        return address;
    }

    public static ContactInformation toContactInformation(
        UserProfile userProfile) 
    {
        Precondition.assertNotNull("userprofile", userProfile);
        ContactInformation contactInfo = new ContactInformation();
        contactInfo.setHomePhone(userProfile.getContactInfoHomePhone());
        return contactInfo;
    }

    public static Gender toGender(GenderEnumeration gender) {
        if (gender == null) { 
            return null;
        }
        if (GenderEnumeration.MALE.equals(gender.getName())) {
            return Gender.Male;
        }
        if (GenderEnumeration.FEMALE.equals(gender.getName())) {
            return Gender.Female;
        }
        return null;
    }
    
    public static ArrayList<Facility> toFacilities(
        Collection<gov.va.med.mhv.usermgmt.transfer.Facility> facilities)
    {
        Precondition.assertNotNull("facilities", facilities);
        ArrayList<Facility> registryFacilities = new ArrayList<Facility>();
        if (facilities != null) {
            for (gov.va.med.mhv.usermgmt.transfer.Facility facility:
                   facilities)
            {
                registryFacilities.add(toFacility(facility));
            }
        }
        return registryFacilities;
    }

    public static Facility toFacility(
        gov.va.med.mhv.usermgmt.transfer.Facility facility)
    {
        Precondition.assertNotNull("facility", facility);
        Facility registryFacility = new Facility();
        // DFN is a currently a placeholder for future extensions
        registryFacility.setDfn(null);
        // We currently do not use the facilityId
        registryFacility.setFacilityId(null);
        // NOTE: name denotes the station number
        registryFacility.setStationNumber(facility.getName());
        return registryFacility;
    }


    public static Status createOkStatus() {
        Status status = new Status();
        status.setStatus(StatusType.Ok);
        status.setStatusDescription(null);
        return status;
    }

    public static Status createErrorStatus(String description) {
        Status status = new Status();
        status.setStatus(StatusType.Error);
        status.setStatusDescription(description);
        return status;
    }

    public static void setFacilities(
        gov.va.med.mhv.usermgmt.transfer.Patient patient,
        List<Facility> facilities)
    {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotNull("facilities", facilities);
        patient.setFacilitys(new HashSet());
        for (Facility registryFacility: facilities) {
            patient.addFacility(fromFacility(registryFacility));
        }
    }

    private static gov.va.med.mhv.usermgmt.transfer.Facility fromFacility(
        Facility registryFacility)
    {
        Precondition.assertNotNull("registryFacility", registryFacility);
        gov.va.med.mhv.usermgmt.transfer.Facility facility =
            TransferObjectFactory.createFacility();
        facility.setName(registryFacility.getStationNumber());
        facility.setFacilityInfo(getFacilityInfo(registryFacility.
            getStationNumber()));
        return facility;
    }

    private static FacilityInfo getFacilityInfo(String stationNumber) {
        FacilityInfo facilityInfo = null;
        if (!StringUtils.isBlank(stationNumber)) {
            List facilityInfoBOs = FacilityInfoBO.queryByStationNumber(
                stationNumber);
            if ((facilityInfoBOs != null) && (facilityInfoBOs.size() > 0)) {
                facilityInfo = ((FacilityInfoBO) facilityInfoBOs.iterator().
                    next()).getFacilityInfoValues();
            }
        }
        return facilityInfo;
    }

    public static void setPersonalInfo(
        gov.va.med.mhv.usermgmt.transfer.Patient patient,
        PersonalInfo personalInfo)
    {
        Precondition.assertNotNull("patient", patient);
        Precondition.assertNotNull("personalInfo", personalInfo);
        UserProfile userProfile = patient.getUserProfile();
        Precondition.assertNotNull("patient.userProfile", userProfile);
        setName(userProfile, personalInfo);
        // unknown in MHV: ignore mothers maiden name
        userProfile.setBirthDate(personalInfo.getDateOfBirth());
        // unknown in MHV: ignore place of birth
        userProfile.setGender(toGender(personalInfo.getGender()));
        userProfile.setSsn(personalInfo.getSsn());
        // unknown in MHV: ignore multiple births indicator
        // unknown in MHV: ignore alias
        // unknown in MHV: ignore etnicity information
        // unknown in MHV: ignore race information
        setAddress(userProfile, personalInfo);
        setContactInformation(userProfile, personalInfo);
    }
    
    public static void setName(UserProfile userProfile, 
        PersonalInfo personalInfo)
    {
        Precondition.assertNotNull("userProfile", userProfile);
        Precondition.assertNotNull("personalInfo", personalInfo);
        Precondition.assertNotNull("personalInfo.name", personalInfo.getName());
        userProfile.setFirstName(personalInfo.getName().getFirstName());
        userProfile.setMiddleName(personalInfo.getName().getMiddleName());
        userProfile.setLastName(personalInfo.getName().getLastName());
        userProfile.setSuffix(SuffixEnumeration.getEnum(personalInfo.getName().
            getSuffix()));
        userProfile.setTitle(TitleEnumeration.getEnum(personalInfo.getName().
            getPrefix()));
    }

    public static void setAddress(UserProfile userProfile, 
        PersonalInfo personalInfo)
    {
        Precondition.assertNotNull("userProfile", userProfile);
        Precondition.assertNotNull("personalInfo", personalInfo);
        if (personalInfo.getPermanentAddress() == null) {
            return;
        }
        Address address = personalInfo.getPermanentAddress();
        userProfile.setAddressStreet1(address.getStreetAddress1());
        userProfile.setAddressStreet2(address.getStreetAddress2());
        userProfile.setAddressCity(address.getCity());
        StateEnumeration addressState = StateEnumeration.getEnum(address.
            getStateOrProvince());
        if (addressState != null) {
            userProfile.setAddressState(addressState);
            userProfile.setAddressProvince(null);
        } else {
            userProfile.setAddressProvince(address.getStateOrProvince());
            userProfile.setAddressState(null);
        }
        userProfile.setAddressPostalCode(address.getZipOrPostalCode());
        userProfile.setAddressCountry(CountryEnumeration.getEnum(address.
            getCountry()));
    }
    
    public static void setContactInformation(UserProfile userProfile, 
        PersonalInfo personalInfo)
    {
        Precondition.assertNotNull("userProfile", userProfile);
        Precondition.assertNotNull("personalInfo", personalInfo);
        if (personalInfo.getContactInformation() == null) {
            return;
        }
        ContactInformation contactInfo = personalInfo.getContactInformation();
        userProfile.setContactInfoHomePhone(contactInfo.getHomePhone());
    }

    public static String describe(Patient patient) {
        StringBuilder builder = new StringBuilder();
        if (patient == null) {
            builder.append("[Patient " + UNKNOWN + "]");
        } else { 
            builder.append("Patient[ICN: ");
            builder.append(patient.getIcn());
            builder.append("; ");
            builder.append(describe(patient.getPersonalInfo(), false));
            builder.append("; ");
            builder.append(describe(patient.getFacilities()));
            builder.append("]");
        }
        return builder.toString();
    }
    
    public static String describe(PersonalInfo personalInfo) {
        return describe(personalInfo, false);
        
    }
    public static String describe(PersonalInfo personalInfo, boolean header) {
        StringBuilder builder = new StringBuilder();
        if (personalInfo == null) {
            builder.append("[PersonalInfo " + UNKNOWN + "]");
        } else {
            builder.append((header) ? "PersonalInfo[": "");
            builder.append("Name: ");
            builder.append(describe(personalInfo.getName(), false));
            builder.append("; Mothers Maiden Name: ");
            builder.append(describe(personalInfo.getMothersMaidenName()));
            builder.append("; DOB: ");
            builder.append(describe(personalInfo.getDateOfBirth()));
            builder.append("; POB: ");
            builder.append(describe(personalInfo.getPlaceOfBirth(), true));
            builder.append("; Gender: ");
            builder.append(describe(personalInfo.getGender()));
            builder.append("; SSN: ");
            builder.append(describeSsn(personalInfo.getSsn()));
            builder.append("; Multiple Birth Indicator: ");
            builder.append(describe(personalInfo.getMultipleBirthIndicator()));
            builder.append("; Alias: ");
            builder.append(describe(personalInfo.getAlias(), true));
            builder.append("; Etnicity: ");
            builder.append(describe(personalInfo.getEtnicity()));
            builder.append("; Race: ");
            builder.append(describe(personalInfo.getRace()));
            builder.append("; PermanentAddress: ");
            builder.append(describe(personalInfo.getPermanentAddress(), true));
            builder.append("; ContactInfo: ");
            builder.append(describe(personalInfo.getContactInformation(), 
                true));
            builder.append((header) ? "]" : "");
        }
        return builder.toString();
    }
    
    public static String describe(Gender gender) {
        return (gender != null) ? gender.getName() : UNKNOWN;
    }

    public static String describe(Name name, boolean header) {
        StringBuilder builder = new StringBuilder();
        if (name == null) {
            builder.append("[Name " + UNKNOWN + "]");
        } else {
            builder.append((header) ? "Name[": "");
            if (StringUtils.isBlank(name.getFirstName())
                && StringUtils.isBlank(name.getMiddleName())
                && StringUtils.isBlank(name.getLastName())
                && StringUtils.isBlank(name.getPrefix()) 
                && StringUtils.isBlank(name.getSuffix())) 
            {
                builder.append(UNKNOWN);
            } else {
                builder.append(StringUtils.defaultString(name.getPrefix()));
                builder.append(StringUtils.defaultString(name.
                    getLastName(), "-"));
                builder.append(", ").append(StringUtils.defaultString(
                    name.getFirstName(), "-"));
                if (!StringUtils.isBlank(name.getMiddleName())) {
                    builder.append(" ").append(name.getMiddleName());
                }
                builder.append(StringUtils.defaultString(name.getSuffix()));
            }
            builder.append((header) ? "]" : "");
        }
        return builder.toString();
    }

    public static String describe(Address address, boolean header) {
        StringBuilder builder = new StringBuilder();
        if (address == null) {
            builder.append("[Address " + UNKNOWN + "]");
        } else {
            builder.append((header) ? "Address[": "");
            builder.append("Street address 1: ");
            builder.append(address.getStreetAddress1());
            builder.append("Street address 2: ");
            builder.append(address.getStreetAddress2());
            builder.append("Street address 3: ");
            builder.append(address.getStreetAddress3());
            builder.append("City: ");
            builder.append(describe(address.getCity()));
            builder.append("State or Provence: ");
            builder.append(describe(address.getStateOrProvince()));
            builder.append("Zip or Postal Code: ");
            builder.append(describe(address.getZipOrPostalCode()));
            builder.append("County Code: ");
            builder.append(describe(address.getCounty()));
            builder.append("Country Code: ");
            builder.append(describe(address.getCountry()));
            builder.append("Bad Address: ");
            builder.append(describe(address.getBadAddressIndicator()));
            builder.append((header) ? "]" : "");
        }
        return builder.toString();
    }

    public static String describe(ContactInformation contactInfo, 
        boolean header) 
    {
        StringBuilder builder = new StringBuilder();
        if (contactInfo == null) {
            builder.append("[ContactInfo " + UNKNOWN + "]");
        } else {
            builder.append((header) ? "ContactInfo[": "");
            builder.append("Home Phone: ");
            builder.append(describe(contactInfo.getHomePhone()));
            builder.append((header) ? "]" : "");
        }
        return builder.toString();
    }


    public static String describe(PlaceOfBirth pob, boolean header) {
        StringBuilder builder = new StringBuilder();
        if (pob == null) {
            builder.append("[PlaceOfBirth " + UNKNOWN + "]");
        } else {
            builder.append((header) ? "PlaceOfBirth[": "");
            builder.append("City: ");
            builder.append(describe(pob.getCity()));
            builder.append("State: ");
            builder.append(describe(pob.getState()));
            builder.append((header) ? "]" : "");
        }
        return builder.toString();
    }

    public static String describe(Alias alias, boolean header) {
        StringBuilder builder = new StringBuilder();
        if (alias == null) {
            builder.append("[Alias " + UNKNOWN + "]");
        } else {
            builder.append((header) ? "Alias[": "");
            builder.append("Name: ");
            builder.append(describe(alias.getName(), false));
            builder.append("Ssn: ");
            builder.append(describeSsn(alias.getSsn()));
            builder.append((header) ? "]" : "");
        }
        return builder.toString();
    }

    public static String describe(List<Facility> facilities) { 
        StringBuilder builder = new StringBuilder();
        builder.append("Facilities {");
        if (facilities != null) {
            for (Facility facility: facilities) {
                builder.append(" ").append(facility.getStationNumber());
            }
            builder.append(" ");
        }
        builder.append("}");
        return builder.toString();
    }

    public static String describe(Status status) {
        StringBuilder builder = new StringBuilder();
        if (status == null) {
            builder.append("[Status " + UNKNOWN + "]");
        } else {
            builder.append("Status[");
            builder.append((status.getStatus() == null) ? UNKNOWN
                : ((StatusType.Ok.equals(status.getStatus())) ? "Ok" 
                    : "Error"));
            if (!StringUtils.isBlank(status.getStatusDescription())) {
                builder.append(": ").append(status.getStatusDescription());
            }
            builder.append("]");
        }
        return builder.toString();
    }

    private static GenderEnumeration toGender(Gender gender) {
        return (gender != null) ? GenderEnumeration.getEnum(gender.getName())
            : null;
    }

    private static String describe(Date date) {
        return describe(date, true);
    }
    private static String describe(Date date, boolean formatted) {
        if (date == null) {
            return UNKNOWN;
        }
        return (formatted) ? DATE_FORMAT.format(date) : date.toString();
    }

    private static String describe(Indicator indicator) {
        return (indicator != null) ? indicator.toString() : UNKNOWN;
    }

    private static String describeSsn(String ssn) {
        if (StringUtils.isBlank(ssn)) {
            return UNKNOWN;
        }
        return "***-**-" + StringUtils.substring(ssn, 7);
    }

    private static String describe(String value) {
        return (value != null) ? value : UNKNOWN;
    }


}
